<?php
require_once 'config.php';

// Helper para leer números (permitiendo coma o punto)
function getNumber($name) {
    $v = $_POST[$name] ?? 0;
    $v = str_replace(['.', ','], ['', '.'], $v); // “1.200.000,50” -> “1200000.50”
    return (float)$v;
}

$datos = [];
$resultados = [];
$errores = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // ===== Campos básicos =====
    $datos['franquicia'] = trim($_POST['franquicia'] ?? '');
    $datos['responsable_nombre'] = trim($_POST['responsable_nombre'] ?? '');
    $datos['responsable_apellidos'] = trim($_POST['responsable_apellidos'] ?? '');
    $datos['mes_analisis'] = trim($_POST['mes_analisis'] ?? '');

    $datos['ventas_mes'] = getNumber('ventas_mes');
    $datos['num_empleados'] = (int)($_POST['num_empleados'] ?? 0);

    // Validaciones simples
    if ($datos['franquicia'] === '') $errores[] = 'El nombre de la franquicia es obligatorio.';
    if ($datos['responsable_nombre'] === '') $errores[] = 'El nombre del responsable es obligatorio.';
    if ($datos['ventas_mes'] <= 0) $errores[] = 'Las ventas del mes deben ser mayores a 0.';
    if ($datos['num_empleados'] <= 0) $errores[] = 'El número de empleados debe ser mayor a 0.';

    // ===== Gastos =====
    $datos['arriendo_admin']        = getNumber('arriendo_admin');
    $datos['nomina']                = getNumber('nomina');
    $datos['servicios_publicos']    = getNumber('servicios_publicos');
    $datos['contador']              = getNumber('contador');
    $datos['calidad']               = getNumber('calidad');
    $datos['papeleria']             = getNumber('papeleria');
    $datos['transporte_parqueadero']= getNumber('transporte_parqueadero');
    $datos['impuestos']             = getNumber('impuestos');
    $datos['mantenimiento']         = getNumber('mantenimiento');
    $datos['mercadeo_activaciones'] = getNumber('mercadeo_activaciones');
    $datos['plataformas']           = getNumber('plataformas');
    $datos['aseo_insumos']          = getNumber('aseo_insumos');
    $datos['otros_gastos']          = getNumber('otros_gastos');
    $datos['credito']               = getNumber('credito');
    $datos['gastos_legales']        = getNumber('gastos_legales');
    $datos['gastos_financieros']    = getNumber('gastos_financieros');

    // Total Gastos (sin crédito ni impuestos según definición de campo Total Gastos)
    $resultados['total_gastos'] =
        $datos['arriendo_admin'] +
        $datos['nomina'] +
        $datos['servicios_publicos'] +
        $datos['contador'] +
        $datos['calidad'] +
        $datos['papeleria'] +
        $datos['transporte_parqueadero'] +
        $datos['mantenimiento'] +
        $datos['mercadeo_activaciones'] +
        $datos['plataformas'] +
        $datos['aseo_insumos'] +
        $datos['otros_gastos'] +
        $datos['gastos_legales'] +
        $datos['gastos_financieros'];

    // ===== Costos =====
    $datos['cafe']           = getNumber('cafe');
    $datos['leche']          = getNumber('leche');
    $datos['frutas_verduras']= getNumber('frutas_verduras');
    $datos['reposteria']     = getNumber('reposteria');
    $datos['otros_costos']   = getNumber('otros_costos');

    $resultados['total_costos'] =
        $datos['cafe'] +
        $datos['leche'] +
        $datos['frutas_verduras'] +
        $datos['reposteria'] +
        $datos['otros_costos'];

    // ===== Inventario =====
    $datos['inventario'] = getNumber('inventario');
    $resultados['inventario_sugerido'] = $datos['ventas_mes'] / 10;

    // ===== Cálculos principales =====
    $resultados['promedio_dia'] = $datos['ventas_mes'] / 30;

    // Porcentaje total costos (costo de producción)
    if ($datos['ventas_mes'] > 0) {
        $resultados['porcentaje_costos'] = ($resultados['total_costos'] / $datos['ventas_mes']) * 100;
    } else {
        $resultados['porcentaje_costos'] = 0;
    }

    // Utilidad Bruta: (Ventas - Costos) + Inventario
    $resultados['utilidad_bruta'] = ($datos['ventas_mes'] - $resultados['total_costos']) + $datos['inventario'];

    // Utilidad Operacional = Utilidad Bruta - Total Gastos
    $resultados['utilidad_operacional'] = $resultados['utilidad_bruta'] - $resultados['total_gastos'];

    // Utilidad Neta = Utilidad Operacional - Crédito - Gastos Financieros - Impuestos
    $resultados['utilidad_neta'] =
        $resultados['utilidad_operacional'] -
        $datos['credito'] -
        $datos['gastos_financieros'] -
        $datos['impuestos'];

    // Utilidad neta - Inventario
    $resultados['utilidad_neta_menos_inventario'] =
        $resultados['utilidad_neta'] - $datos['inventario'];

    // Utilidad neta - inventario / ventas totales (%)
    if ($datos['ventas_mes'] > 0) {
        $resultados['utilidad_neta_menos_inv_sobre_ventas'] =
            ($resultados['utilidad_neta_menos_inventario'] / $datos['ventas_mes']) * 100;
    } else {
        $resultados['utilidad_neta_menos_inv_sobre_ventas'] = 0;
    }

    // ===== Resultados esperados / porcentajes =====
    // Costo de producción (igual a porcentaje_costos)
    $resultados['costo_produccion_pct'] = $resultados['porcentaje_costos'];

    if ($datos['ventas_mes'] > 0) {
        $resultados['part_arriendo_pct']   = ($datos['arriendo_admin']     / $datos['ventas_mes']) * 100;
        $resultados['part_servicios_pct']  = ($datos['servicios_publicos'] / $datos['ventas_mes']) * 100;
        $resultados['part_nomina_pct']     = ($datos['nomina']             / $datos['ventas_mes']) * 100;
        $resultados['margen_neto_pct']     = ($resultados['utilidad_neta'] / $datos['ventas_mes']) * 100;
        $resultados['margen_operacional_pct'] =
            ($resultados['utilidad_operacional'] / $datos['ventas_mes']) * 100;
    } else {
        $resultados['part_arriendo_pct'] = $resultados['part_servicios_pct'] =
        $resultados['part_nomina_pct'] = $resultados['margen_neto_pct'] =
        $resultados['margen_operacional_pct'] = 0;
    }

    if ($resultados['total_costos'] > 0) {
        $resultados['part_cafe_pct']        = ($datos['cafe']            / $resultados['total_costos']) * 100;
        $resultados['part_leche_pct']       = ($datos['leche']           / $resultados['total_costos']) * 100;
        $resultados['part_frutas_pct']      = ($datos['frutas_verduras'] / $resultados['total_costos']) * 100;
        $resultados['part_reposteria_pct']  = ($datos['reposteria']      / $resultados['total_costos']) * 100;
        $resultados['part_otros_costos_pct']= ($datos['otros_costos']    / $resultados['total_costos']) * 100;
    } else {
        $resultados['part_cafe_pct'] =
        $resultados['part_leche_pct'] =
        $resultados['part_frutas_pct'] =
        $resultados['part_reposteria_pct'] =
        $resultados['part_otros_costos_pct'] = 0;
    }

    // ===== Guardar en base de datos si no hay errores =====
    if (empty($errores)) {
        $stmt = $pdo->prepare("
          INSERT INTO proyecciones (
            franquicia, responsable_nombre, responsable_apellidos, mes_analisis,
            ventas_mes, num_empleados, promedio_dia,
            arriendo_admin, nomina, servicios_publicos, contador, calidad, papeleria,
            transporte_parqueadero, impuestos, mantenimiento, mercadeo_activaciones,
            plataformas, aseo_insumos, otros_gastos, credito, gastos_legales,
            gastos_financieros, total_gastos,
            cafe, leche, frutas_verduras, reposteria, otros_costos, total_costos,
            porcentaje_costos, inventario, inventario_sugerido,
            utilidad_bruta, utilidad_operacional, utilidad_neta,
            utilidad_neta_menos_inventario, utilidad_neta_menos_inv_sobre_ventas,
            costo_produccion_pct, part_arriendo_pct, part_servicios_pct,
            part_nomina_pct, margen_neto_pct, margen_operacional_pct,
            part_cafe_pct, part_leche_pct, part_frutas_pct,
            part_reposteria_pct, part_otros_costos_pct
          ) VALUES (
            :franquicia, :responsable_nombre, :responsable_apellidos, :mes_analisis,
            :ventas_mes, :num_empleados, :promedio_dia,
            :arriendo_admin, :nomina, :servicios_publicos, :contador, :calidad, :papeleria,
            :transporte_parqueadero, :impuestos, :mantenimiento, :mercadeo_activaciones,
            :plataformas, :aseo_insumos, :otros_gastos, :credito, :gastos_legales,
            :gastos_financieros, :total_gastos,
            :cafe, :leche, :frutas_verduras, :reposteria, :otros_costos, :total_costos,
            :porcentaje_costos, :inventario, :inventario_sugerido,
            :utilidad_bruta, :utilidad_operacional, :utilidad_neta,
            :utilidad_neta_menos_inventario, :utilidad_neta_menos_inv_sobre_ventas,
            :costo_produccion_pct, :part_arriendo_pct, :part_servicios_pct,
            :part_nomina_pct, :margen_neto_pct, :margen_operacional_pct,
            :part_cafe_pct, :part_leche_pct, :part_frutas_pct,
            :part_reposteria_pct, :part_otros_costos_pct
          )
        ");

        $stmt->execute(array_merge($datos, $resultados));
    }
}

// Helper para formatear moneda y porcentaje
function m($n) { return '$' . number_format($n, 0, ',', '.'); }
function p($n) { return number_format($n, 2, ',', '.') . '%'; }

// Para gráficas: si tenemos resultados
$hayResultados = $_SERVER['REQUEST_METHOD'] === 'POST' && empty($errores);
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Calculadora de Proyecciones - Nativos</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- Bootstrap CDN -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <!-- Chart.js -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>

  <style>
    body {
      background: #f5f7fb;
    }
    .card {
      border-radius: 16px;
      box-shadow: 0 10px 25px rgba(0,0,0,0.05);
      border: none;
    }
    .section-title {
      border-left: 4px solid #28a745;
      padding-left: 10px;
      font-weight: 600;
      margin-top: 20px;
      margin-bottom: 10px;
    }
    .badge-critico { background:#dc3545; }
    .badge-bien { background:#28a745; }
    .badge-alerta { background:#ffc107; color:#000; }

    @media print {
      body { background: #fff; }
      .no-print { display:none !important; }
      .card { box-shadow:none; border:none; }
      .container { max-width: 100%; }
    }
  </style>
</head>
<body>
<div class="container py-4">

  <div class="d-flex justify-content-between align-items-center mb-4 no-print">
    <h1 class="h3 mb-0">Calculadora de Proyecciones - Nativos</h1>
    <button class="btn btn-outline-secondary" onclick="window.print()">Imprimir / Guardar en PDF</button>
  </div>

  <?php if (!empty($errores)): ?>
    <div class="alert alert-danger">
      <ul class="mb-0">
        <?php foreach ($errores as $e): ?>
          <li><?= htmlspecialchars($e) ?></li>
        <?php endforeach; ?>
      </ul>
    </div>
  <?php endif; ?>

  <!-- ===== FORMULARIO ===== -->
  <div class="card mb-4 no-print" id="formulario">
    <div class="card-body">
      <form method="post">
        <div class="section-title">Datos generales</div>
        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">Nombre Franquicia Nativos *</label>
            <input type="text" name="franquicia" class="form-control" required
                   value="<?= htmlspecialchars($datos['franquicia'] ?? '') ?>">
          </div>
          <div class="col-md-4">
            <label class="form-label">Nombre Responsable *</label>
            <input type="text" name="responsable_nombre" class="form-control" required
                   value="<?= htmlspecialchars($datos['responsable_nombre'] ?? '') ?>">
          </div>
          <div class="col-md-4">
            <label class="form-label">Apellidos Responsable *</label>
            <input type="text" name="responsable_apellidos" class="form-control" required
                   value="<?= htmlspecialchars($datos['responsable_apellidos'] ?? '') ?>">
          </div>
          <div class="col-md-4">
            <label class="form-label">Mes de análisis</label>
            <input type="text" name="mes_analisis" class="form-control"
                   placeholder="Ej: Septiembre 2025"
                   value="<?= htmlspecialchars($datos['mes_analisis'] ?? '') ?>">
          </div>
          <div class="col-md-4">
            <label class="form-label">Ventas del mes a analizar *</label>
            <input type="number" step="0.01" min="0" name="ventas_mes" class="form-control" required
                   value="<?= htmlspecialchars($datos['ventas_mes'] ?? '') ?>">
          </div>
          <div class="col-md-4">
            <label class="form-label">Número de empleados *</label>
            <input type="number" min="1" name="num_empleados" class="form-control" required
                   value="<?= htmlspecialchars($datos['num_empleados'] ?? '') ?>">
          </div>
        </div>

        <hr class="my-4">

        <div class="section-title">Gastos operacionales</div>
        <div class="row g-3">
          <div class="col-md-3">
            <label class="form-label">Arriendo + Administración *</label>
            <input type="number" step="0.01" min="0" name="arriendo_admin" class="form-control" required
                   value="<?= htmlspecialchars($datos['arriendo_admin'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Nómina (con SS) *</label>
            <input type="number" step="0.01" min="0" name="nomina" class="form-control" required
                   value="<?= htmlspecialchars($datos['nomina'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Servicios Públicos *</label>
            <input type="number" step="0.01" min="0" name="servicios_publicos" class="form-control" required
                   value="<?= htmlspecialchars($datos['servicios_publicos'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Contador *</label>
            <input type="number" step="0.01" min="0" name="contador" class="form-control" required
                   value="<?= htmlspecialchars($datos['contador'] ?? '') ?>">
          </div>

          <div class="col-md-3">
            <label class="form-label">Calidad *</label>
            <input type="number" step="0.01" min="0" name="calidad" class="form-control" required
                   value="<?= htmlspecialchars($datos['calidad'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Papelería *</label>
            <input type="number" step="0.01" min="0" name="papeleria" class="form-control" required
                   value="<?= htmlspecialchars($datos['papeleria'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Transporte y parqueadero *</label>
            <input type="number" step="0.01" min="0" name="transporte_parqueadero" class="form-control" required
                   value="<?= htmlspecialchars($datos['transporte_parqueadero'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Impuestos (Ind. y Com., Hacienda) *</label>
            <input type="number" step="0.01" min="0" name="impuestos" class="form-control" required
                   value="<?= htmlspecialchars($datos['impuestos'] ?? '') ?>">
          </div>

          <div class="col-md-3">
            <label class="form-label">Mantenimiento *</label>
            <input type="number" step="0.01" min="0" name="mantenimiento" class="form-control" required
                   value="<?= htmlspecialchars($datos['mantenimiento'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Mercadeo y activaciones *</label>
            <input type="number" step="0.01" min="0" name="mercadeo_activaciones" class="form-control" required
                   value="<?= htmlspecialchars($datos['mercadeo_activaciones'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Plataformas (POS y FE) *</label>
            <input type="number" step="0.01" min="0" name="plataformas" class="form-control" required
                   value="<?= htmlspecialchars($datos['plataformas'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Aseo (insumos y materiales) *</label>
            <input type="number" step="0.01" min="0" name="aseo_insumos" class="form-control" required
                   value="<?= htmlspecialchars($datos['aseo_insumos'] ?? '') ?>">
          </div>

          <div class="col-md-3">
            <label class="form-label">Otros gastos *</label>
            <input type="number" step="0.01" min="0" name="otros_gastos" class="form-control" required
                   value="<?= htmlspecialchars($datos['otros_gastos'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Crédito *</label>
            <input type="number" step="0.01" min="0" name="credito" class="form-control" required
                   value="<?= htmlspecialchars($datos['credito'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Gastos legales *</label>
            <input type="number" step="0.01" min="0" name="gastos_legales" class="form-control" required
                   value="<?= htmlspecialchars($datos['gastos_legales'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Gastos financieros *</label>
            <input type="number" step="0.01" min="0" name="gastos_financieros" class="form-control" required
                   value="<?= htmlspecialchars($datos['gastos_financieros'] ?? '') ?>">
          </div>
        </div>

        <hr class="my-4">

        <div class="section-title">Costos de producción</div>
        <div class="row g-3">
          <div class="col-md-2">
            <label class="form-label">Café *</label>
            <input type="number" step="0.01" min="0" name="cafe" class="form-control" required
                   value="<?= htmlspecialchars($datos['cafe'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Leche *</label>
            <input type="number" step="0.01" min="0" name="leche" class="form-control" required
                   value="<?= htmlspecialchars($datos['leche'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Frutas y verduras *</label>
            <input type="number" step="0.01" min="0" name="frutas_verduras" class="form-control" required
                   value="<?= htmlspecialchars($datos['frutas_verduras'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Repostería *</label>
            <input type="number" step="0.01" min="0" name="reposteria" class="form-control" required
                   value="<?= htmlspecialchars($datos['reposteria'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Otros costos *</label>
            <input type="number" step="0.01" min="0" name="otros_costos" class="form-control" required
                   value="<?= htmlspecialchars($datos['otros_costos'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Inventario actual *</label>
            <input type="number" step="0.01" min="0" name="inventario" class="form-control" required
                   value="<?= htmlspecialchars($datos['inventario'] ?? '') ?>">
          </div>
        </div>

        <div class="mt-4 text-end">
          <button type="submit" class="btn btn-success">Calcular proyección</button>
        </div>
      </form>
    </div>
  </div>

  <!-- ===== REPORTE ===== -->
  <?php if ($hayResultados): ?>
    <div class="card mb-4" id="reporte">
      <div class="card-body">
        <h2 class="h4 mb-3">Informe de Proyección - <?= htmlspecialchars($datos['franquicia']) ?></h2>
        <p class="text-muted mb-4">
          Mes de análisis: <strong><?= htmlspecialchars($datos['mes_analisis'] ?: 'No especificado') ?></strong>
        </p>

        <!-- Estado de resultados resumido -->
        <div class="row g-3 mb-4">
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Ventas del mes</div>
              <div class="fw-bold fs-5"><?= m($datos['ventas_mes']) ?></div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Promedio ventas día</div>
              <div class="fw-bold fs-5"><?= m($resultados['promedio_dia']) ?></div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Total costos</div>
              <div class="fw-bold fs-5"><?= m($resultados['total_costos']) ?></div>
              <div class="small">
                Costo producción:
                <?php
                  $badge = $resultados['costo_produccion_pct'] > 30 ? 'badge-critico' : 'badge-bien';
                ?>
                <span class="badge <?= $badge ?>"><?= p($resultados['costo_produccion_pct']) ?></span>
              </div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Total gastos (operacionales)</div>
              <div class="fw-bold fs-5"><?= m($resultados['total_gastos']) ?></div>
            </div>
          </div>
        </div>

        <div class="row g-3 mb-4">
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Utilidad bruta</div>
              <div class="fw-bold fs-5"><?= m($resultados['utilidad_bruta']) ?></div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Utilidad operacional</div>
              <div class="fw-bold fs-5"><?= m($resultados['utilidad_operacional']) ?></div>
              <?php
                $badge = $resultados['margen_operacional_pct'] >= 18 ? 'badge-bien' : 'badge-critico';
              ?>
              <span class="badge <?= $badge ?> mt-1">
                Margen operacional: <?= p($resultados['margen_operacional_pct']) ?>
              </span>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Utilidad neta</div>
              <div class="fw-bold fs-5"><?= m($resultados['utilidad_neta']) ?></div>
              <?php
                $badge = $resultados['margen_neto_pct'] >= 16 ? 'badge-bien' : 'badge-critico';
              ?>
              <span class="badge <?= $badge ?> mt-1">
                Margen neto: <?= p($resultados['margen_neto_pct']) ?>
              </span>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Utilidad neta - inventario</div>
              <div class="fw-bold fs-5"><?= m($resultados['utilidad_neta_menos_inventario']) ?></div>
              <div class="small">
                Sobre ventas:
                <strong><?= p($resultados['utilidad_neta_menos_inv_sobre_ventas']) ?></strong>
              </div>
            </div>
          </div>
        </div>

        <!-- Inventario -->
        <div class="mb-4">
          <div class="section-title">Inventario</div>
          <p>
            Inventario actual: <strong><?= m($datos['inventario']) ?></strong><br>
            Inventario sugerido (10% de las ventas): <strong><?= m($resultados['inventario_sugerido']) ?></strong>
          </p>
        </div>

        <!-- Gráficas -->
        <div class="row g-4 mb-4">
          <div class="col-md-6">
            <div class="section-title">Resumen financiero</div>
            <canvas id="chartResumen"></canvas>
          </div>
          <div class="col-md-6">
            <div class="section-title">Distribución de costos</div>
            <canvas id="chartCostos"></canvas>
          </div>
        </div>

        <!-- Texto tipo carta -->
        <div class="mt-4">
          <div class="section-title">Informe detallado</div>
          <p>
            Un cordial saludo. <strong><?= htmlspecialchars($datos['franquicia']) ?></strong>
          </p>
          <p>
            Espero que todo esté bien.
            <strong><?= htmlspecialchars($datos['responsable_nombre'] . ' ' . $datos['responsable_apellidos']) ?></strong>
          </p>
          <p>
            Gracias por cumplir con el compromiso del envío del informe mensual de gastos, costos y ventas.
            Esto nos permitirá crecer juntos.
          </p>

          <ol>
            <li>
              <strong>Ventas (<?= htmlspecialchars($datos['franquicia']) ?>)</strong><br>
              Ventas (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>): <?= m($datos['ventas_mes']) ?><br>
              Promedio/día (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>):
              <?= m($resultados['promedio_dia']) ?>
            </li>
            <br>
            <li>
              <strong>Costos de producción</strong><br>
              Costos (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>): <?= m($resultados['total_costos']) ?>
              (<?= p($resultados['porcentaje_costos']) ?>)<br>
              No pueden superar el 30% y están en el <?= p($resultados['porcentaje_costos']) ?>.
            </li>
            <br>
            <li>
              <strong>Inventario</strong><br>
              Debería estar en <?= m($resultados['inventario_sugerido']) ?>.
              El inventario está en <?= m($datos['inventario']) ?>.
            </li>
            <br>
            <li>
              <strong>Gastos operacionales (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>)</strong><br>
              Gastos: <?= m($resultados['total_gastos']) ?>
            </li>
          </ol>

          <hr>

          <h5>Estado de Resultados (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>)</h5>
          <p>
            Ventas: <?= m($datos['ventas_mes']) ?><br>
            Costos: <?= m($resultados['total_costos']) ?> (<?= p($resultados['porcentaje_costos']) ?>)<br>
            Inventario: <?= m($datos['inventario']) ?><br>
            Utilidad bruta: <?= m($resultados['utilidad_bruta']) ?><br><br>

            Gastos: <?= m($resultados['total_gastos']) ?><br>
            Utilidad operacional: <?= m($resultados['utilidad_operacional']) ?><br>
            Margen operacional: <?= p($resultados['margen_operacional_pct']) ?><br><br>

            Menos crédito: <?= m($datos['credito']) ?><br>
            Menos gastos financieros: <?= m($datos['gastos_financieros']) ?><br>
            Menos impuestos: <?= m($datos['impuestos']) ?><br><br>

            Utilidad neta: <?= m($resultados['utilidad_neta']) ?><br>
            Margen neto: <?= p($resultados['margen_neto_pct']) ?><br><br>

            La utilidad neta con el actual costo del <?= p($resultados['porcentaje_costos']) ?>,
            luego de cancelar gastos financieros, crédito y tributarios y sin inventario:
            <?= m($datos['inventario']) ?>, es
            <?= m($resultados['utilidad_neta_menos_inventario']) ?>,
            que representa el <?= p($resultados['utilidad_neta_menos_inv_sobre_ventas']) ?>.
            Esta fue la ganancia real percibida por ustedes.
          </p>

          <p class="mt-3">
            Gracias.
          </p>
        </div>

      </div>
    </div>
  <?php endif; ?>

</div>

<?php if ($hayResultados): ?>
<script>
  // Datos PHP -> JS
  const ventas    = <?= json_encode($datos['ventas_mes']) ?>;
  const totalCostos = <?= json_encode($resultados['total_costos']) ?>;
  const totalGastos = <?= json_encode($resultados['total_gastos']) ?>;
  const uBruta   = <?= json_encode($resultados['utilidad_bruta']) ?>;
  const uOper    = <?= json_encode($resultados['utilidad_operacional']) ?>;
  const uNeta    = <?= json_encode($resultados['utilidad_neta']) ?>;

  const costosDetalle = {
    cafe: <?= json_encode($datos['cafe']) ?>,
    leche: <?= json_encode($datos['leche']) ?>,
    frutas: <?= json_encode($datos['frutas_verduras']) ?>,
    reposteria: <?= json_encode($datos['reposteria']) ?>,
    otros: <?= json_encode($datos['otros_costos']) ?>
  };

  // Gráfico resumen financiero (barras)
  const ctxRes = document.getElementById('chartResumen');
  new Chart(ctxRes, {
    type: 'bar',
    data: {
      labels: ['Ventas', 'Total Costos', 'Total Gastos', 'Utilidad Bruta', 'Utilidad Operacional', 'Utilidad Neta'],
      datasets: [{
        label: 'Valor ($)',
        data: [ventas, totalCostos, totalGastos, uBruta, uOper, uNeta]
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { display: false },
        tooltip: {
          callbacks: {
            label: function(ctx) {
              const v = ctx.parsed.y || 0;
              return ' $' + v.toLocaleString('es-CO');
            }
          }
        }
      },
      scales: {
        y: {
          beginAtZero: true
        }
      }
    }
  });

  // Gráfico distribución de costos (pie)
  const ctxCostos = document.getElementById('chartCostos');
  new Chart(ctxCostos, {
    type: 'pie',
    data: {
      labels: ['Café', 'Leche', 'Frutas y verduras', 'Repostería', 'Otros'],
      datasets: [{
        data: [
          costosDetalle.cafe,
          costosDetalle.leche,
          costosDetalle.frutas,
          costosDetalle.reposteria,
          costosDetalle.otros
        ]
      }]
    },
    options: {
      responsive: true,
      plugins: {
        tooltip: {
          callbacks: {
            label: function(ctx) {
              const total = ctx.dataset.data.reduce((a, b) => a + b, 0);
              const val = ctx.parsed || 0;
              const pct = total > 0 ? (val / total * 100) : 0;
              return ctx.label + ': $' + val.toLocaleString('es-CO') +
                ' (' + pct.toFixed(1) + '%)';
            }
          }
        }
      }
    }
  });
</script>
<?php endif; ?>

</body>
</html>
