<?php
require_once __DIR__ . '/auth.php';
require_login();
global $pdo;

// Años disponibles
$stmtYears = $pdo->query("SELECT DISTINCT YEAR(fecha_reporte) AS anio FROM proyecciones ORDER BY anio DESC");
$years = $stmtYears->fetchAll(PDO::FETCH_COLUMN);

$anioSel = $years ? (int)$years[0] : (int)date('Y');

if (isset($_GET['anio']) && ctype_digit($_GET['anio'])) {
    $anioSel = (int)$_GET['anio'];
}

// Franquicias disponibles
$stmtFran = $pdo->query("SELECT DISTINCT franquicia FROM proyecciones ORDER BY franquicia ASC");
$franquicias = $stmtFran->fetchAll(PDO::FETCH_COLUMN);

$franSel = $franquicias[0] ?? '';
if (isset($_GET['franquicia']) && $_GET['franquicia'] !== '') {
    $franSel = $_GET['franquicia'];
}

$rows = [];
if ($years && $franquicias) {
    $sql = "SELECT
              mes_analisis,
              fecha_reporte,
              ventas_mes,
              total_costos,
              total_gastos,
              utilidad_neta,
              margen_neto_pct
            FROM proyecciones
            WHERE YEAR(fecha_reporte) = :anio
              AND franquicia = :fran
            ORDER BY fecha_reporte ASC";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':anio' => $anioSel,
        ':fran' => $franSel,
    ]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function m($n) { return '$' . number_format($n, 0, ',', '.'); }
function p($n) { return number_format($n, 2, ',', '.') . '%'; }
function fechaBonita($fecha) {
    if (!$fecha) return '';
    try {
        $dt = new DateTime($fecha);
        return $dt->format('d/m/Y');
    } catch (Exception $e) {
        return htmlspecialchars($fecha);
    }
}

// Preparar datos para Chart.js
$labels = [];
$dataVentas = [];
$dataUtilNeta = [];
foreach ($rows as $r) {
    $labels[] = $r['mes_analisis'];
    $dataVentas[] = (float)$r['ventas_mes'];
    $dataUtilNeta[] = (float)$r['utilidad_neta'];
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Análisis anual - Calculadora de Proyecciones</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
  <style>
    body { background:#f5f7fb; }
    .card {
      border-radius: 16px;
      box-shadow: 0 10px 25px rgba(0,0,0,0.05);
      border: none;
    }
    .section-title {
      border-left: 4px solid #28a745;
      padding-left: 10px;
      font-weight: 600;
      margin-top: 20px;
      margin-bottom: 10px;
    }
    @media print {
      .no-print { display:none !important; }
      body { background:#fff; }
    }
  </style>
</head>
<body>
<div class="container py-4">

  <nav class="navbar navbar-expand-lg navbar-light bg-white rounded shadow-sm mb-4 no-print">
    <div class="container-fluid">
      <a class="navbar-brand fw-semibold" href="index.php">Calculadora de Proyecciones</a>
      <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navMain">
        <span class="navbar-toggler-icon"></span>
      </button>
      <div class="collapse navbar-collapse" id="navMain">
        <ul class="navbar-nav me-auto mb-2 mb-lg-0">
          <li class="nav-item">
            <a class="nav-link" href="index.php">Nuevo / Editar reporte</a>
          </li>
          <li class="nav-item">
            <a class="nav-link" href="reportes.php">Listado de reportes</a>
          </li>
          <li class="nav-item">
            <a class="nav-link active" aria-current="page" href="analisis.php">Análisis anual</a>
          </li>
        </ul>
        <span class="navbar-text me-3 small">
          <?= htmlspecialchars($_SESSION['user_name'] ?? $_SESSION['user'] ?? '') ?>
        </span>
        <a href="logout.php" class="btn btn-outline-danger btn-sm">Cerrar sesión</a>
      </div>
    </div>
  </nav>

  <div class="card mb-4">
    <div class="card-body">
      <div class="d-flex justify-content-between align-items-center mb-3">
        <h1 class="h5 mb-0">Análisis comparativo anual</h1>
        <button class="btn btn-outline-secondary btn-sm no-print" onclick="window.print()">Imprimir / PDF</button>
      </div>

      <?php if (!$years || !$franquicias): ?>
        <p class="text-muted">Aún no hay información suficiente para el análisis anual.</p>
      <?php else: ?>
        <form method="get" class="row g-3 align-items-end mb-3 no-print">
          <div class="col-md-3">
            <label class="form-label">Año</label>
            <select name="anio" class="form-select">
              <?php foreach ($years as $y): ?>
                <option value="<?= (int)$y ?>" <?= (int)$y === (int)$anioSel ? 'selected' : '' ?>>
                  <?= (int)$y ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-5">
            <label class="form-label">Franquicia</label>
            <select name="franquicia" class="form-select">
              <?php foreach ($franquicias as $f): ?>
                <option value="<?= htmlspecialchars($f) ?>" <?= $f === $franSel ? 'selected' : '' ?>>
                  <?= htmlspecialchars($f) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-2">
            <button type="submit" class="btn btn-success w-100">Ver análisis</button>
          </div>
        </form>

        <p class="mb-3">
          Franquicia: <strong><?= htmlspecialchars($franSel) ?></strong><br>
          Año: <strong><?= (int)$anioSel ?></strong>
        </p>

        <?php if (!$rows): ?>
          <p class="text-muted">No hay reportes para esta combinación de año y franquicia.</p>
        <?php else: ?>
          <div class="row g-4 mb-4">
            <div class="col-md-12">
              <div class="section-title">Evolución de ventas y utilidad neta</div>
              <canvas id="chartAnual"></canvas>
            </div>
          </div>

          <div class="section-title">Detalle mensual</div>
          <div class="table-responsive">
            <table class="table table-sm align-middle">
              <thead>
                <tr>
                  <th>Fecha</th>
                  <th>Mes</th>
                  <th>Ventas</th>
                  <th>Total costos</th>
                  <th>Total gastos</th>
                  <th>Utilidad neta</th>
                  <th>Margen neto</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($rows as $r): ?>
                  <tr>
                    <td><?= fechaBonita($r['fecha_reporte']) ?></td>
                    <td><?= htmlspecialchars($r['mes_analisis']) ?></td>
                    <td><?= m($r['ventas_mes']) ?></td>
                    <td><?= m($r['total_costos']) ?></td>
                    <td><?= m($r['total_gastos']) ?></td>
                    <td><?= m($r['utilidad_neta']) ?></td>
                    <td><?= p($r['margen_neto_pct']) ?></td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>
      <?php endif; ?>
    </div>
  </div>

</div>

<?php if ($rows): ?>
<script>
  const labels = <?= json_encode($labels, ensure_ascii=False) ?>;
  const dataVentas = <?= json_encode($dataVentas) ?>;
  const dataUtilidad = <?= json_encode($dataUtilNeta) ?>;

  const ctx = document.getElementById('chartAnual');
  new Chart(ctx, {
    type: 'bar',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'Ventas',
          data: dataVentas
        },
        {
          label: 'Utilidad neta',
          data: dataUtilidad
        }
      ]
    },
    options: {
      responsive: true,
      plugins: {
        tooltip: {
          callbacks: {
            label: function(ctx) {
              const v = ctx.parsed.y || 0;
              return ctx.dataset.label + ': $' + v.toLocaleString('es-CO');
            }
          }
        }
      },
      scales: {
        y: {
          beginAtZero: true
        }
      }
    }
  });
</script>
<?php endif; ?>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
